$(window).resize(function(){
    responsive_resize();
});

$(function(){
	

	FastClick.attach(document.body);

	$('.collapsible').on('show.bs.collapse', function () {
		$(this).find('.collapse-icon i').removeClass('fa-plus').addClass('fa-minus');	  	
	});

	$('.collapsible').on('hide.bs.collapse', function () {
	  	$(this).find('.collapse-icon i').removeClass('fa-minus').addClass('fa-plus');
	});

	responsive_resize();
	var daApp = new $.da($("#daApp"));
});

/*
--------------------------------------------------------------
APP OBJECTS
--------------------------------------------------------------
*/
(function ($){
	"use scrict"
	$.da = function( element ){
		this.$element = $( element ); // top-level element
		this.init();
	};

	$.da.prototype = {
		init: function() {
			this.$processModal = this.$element.find('#loader');
			this.$msgModal = this.$element.find('#msgModal');
			this.$changeColor = this.$element.find('#changeColor');
			this.$hmCan = this.$element.find('#heatmap');
			this.$hmCanvas = document.getElementById('heatmap');
			this.$hmCtx = this.$hmCanvas.getContext('2d');
			this.$ovCan = this.$element.find('#overlay');
			this.$ovCanvas = document.getElementById('overlay');
			this.$ovCtx = this.$ovCanvas.getContext('2d');
			this.$heatmapBox = this.$element.find('#heatmapBox');
			this.$heatmapBtn = this.$element.find('#heatmapBtn');
			this.$showBoardBtn = this.$element.find('#showDartboard');
			this.$showOptBtn = this.$element.find('#showOptimumPlace');
			this.$progressBar = this.$element.find('#progressBar');
			this.$progressText = this.$element.find('#progressText');
			this.$statsNotes = this.$element.find('#statsNotes span');
			this.$maxExpectedScore = this.$element.find('#maxExpectedScore span');
			this.$aimFor = this.$element.find('#aimFor span');
			this.$sigma = this.$element.find('#sigma span');
			this.$legend = this.$element.find('#legend');

			this.$classic = [new Color({red:255,green:1,blue:1}),new Color({red:255,green:113,blue:1}),new Color({red:255,green:255,blue:1}),new Color({red:255,green:255,blue:255})];
			this.$grayscale = [new Color({red:0,green:0,blue:0}),new Color({red:255,green:255,blue:255})];
			this.$rainbow = [
				new Color({red:255,green:0,blue:0}),
				new Color({red:255,green:219,blue:0}),
				new Color({red:73,green:255,blue:0}),
				new Color({red:0,green:255,blue:146}),
				new Color({red:0,green:146,blue:255}),
				new Color({red:73,green:0,blue:255}),
				new Color({red:255,green:0,blue:219})
			];
			this.$colorSet = this.$classic;
			this.$numColors = "";
			this.$heatColors = [];
			this.$colorScheme = [];

			this.redraw();
			this.addListener();			
		},

		addListener: function() {
			$(window).on("resize",$.proxy(this.redraw,this));
			this.$heatmapBtn.on('click',$.proxy(this.createHeatMap,this));
			this.$showBoardBtn.on('change',$.proxy(this.boardToggle,this));
			this.$showOptBtn.on('change',$.proxy(this.optToggle,this));
			this.$changeColor.on('click',$.proxy(this.changeColor,this));
		},

		setCanvasas: function() {
			$('#heatmapBox').height($('#heatmapBox').width());
 			$('canvas').attr('width',($('#heatmapBox').width()+'px'));
 			$('canvas').attr('height',($('#heatmapBox').height()+'px'));
		},

		drawBoard: function(context, canvas, dim, displayNumbers, margin) {
			var dartboardnumbers = [6,13,4,18,1,20,5,12,9,14,11,8,16,7,19,3,17,2,15,10];
			context.textAlign = "center"
			context.fillStyle = "#000";
			context.font = dim.width/25 + "px Helvetica";
			//set metrics
			var db = {};
			db.x = dim.width/2; // db.x is the central x coordinate.
			db.y = dim.height/2; // db.y is the central y coordinate.
			db.rad = dim.width/2;
			//determines whether a margin is included in the dartboard drawing
			var scaleFactor = (margin ? 172 : 170);
			
			//treble ring
			this.drawCircle(context, db.x, db.y, db.rad*99/scaleFactor);
			this.drawCircle(context, db.x, db.y, db.rad*107/scaleFactor);
			//double ring
			this.drawCircle(context, db.x, db.y, db.rad*170/scaleFactor);
			this.drawCircle(context, db.x, db.y, db.rad*162/scaleFactor);
			//bull
			this.drawCircle(context, db.x, db.y, db.rad*6.35/scaleFactor);
		  	this.drawCircle(context, db.x, db.y, db.rad*15.9/scaleFactor);

			var theta, innerX, innerY, outerX, outerY;

			for (var i = 0; i < 20; i++) {
				theta = Math.PI/20 + i*(Math.PI/10);
				innerX = Math.round((db.rad*15.9/scaleFactor) * Math.cos(theta) + db.x);
				innerY = Math.round(db.y - ((db.rad*15.9/scaleFactor) * Math.sin(theta)));
				outerX = Math.round(((db.rad*170/scaleFactor) * Math.cos(theta)) + db.x);
				outerY = Math.round(db.y - ((db.rad*170/scaleFactor) * Math.sin(theta)));
				
				context.beginPath();
		   		context.moveTo(innerX, innerY);
		  		context.lineTo(outerX, outerY);

			  	if(displayNumbers) {
			  		theta -= Math.PI/20;
			  		outerX = Math.round(((db.rad*135/scaleFactor) * Math.cos(theta)) + db.x);
					outerY = Math.round((db.y+8) - ((db.rad*140/scaleFactor) * Math.sin(theta)));
			  		context.fillText(dartboardnumbers[i], outerX, outerY);
				}
				context.stroke();
		  	}
		},

		drawCircle: function(ctx, x, y, rad) {
			ctx.beginPath();
			ctx.arc(x, y, rad, 0, Math.PI*2, true);
			ctx.stroke();
			ctx.closePath();
		},

		redraw: function() {
 			$('#heatmapBox').height($('#heatmapBox').width());
 			$('#overlay').attr('width',($('#heatmapBox').width()+'px'));
 			$('#overlay').attr('height',($('#heatmapBox').height()+'px'));
 			this.drawBoard(this.$ovCtx,this.$ovCanvas,{'width':this.$heatmapBox.width(),'height':this.$heatmapBox.height()}, true, true);
		},

		createHeatMap: function() {
			var scores = this.$element.find($('#scores')).val();
			
			if (scores==null || scores=="") {
				this.$msgModal.find('.modal-title').text("Error");
				this.$msgModal.find('.modal-body').text("Please enter some scores");
				this.$msgModal.modal("show");
				return false;
			} else {
				var scoresArray = scores.split(',');
				var n = scoresArray.length;
				scores = [];
				try {
					for (var i=0; i<n; i++) {
						scores[i] = parseInt(scoresArray[i].trim());
					}
				} catch (e){
					this.$msgModal.find('.modal-title').text("Error");
					this.$msgModal.find('.modal-body').text("There is a problem with the format of your scores.");
					this.$msgModal.modal("show");
					return false;
				}
			}
			
			var invalidNumbers = this._checkNumbers(scores);
			if (invalidNumbers.length > 0) {
				this.$msgModal.find('.modal-title').text("Error");
				this.$msgModal.find('.modal-body').text("The following scores are not valid: "+invalidNumbers.join(", "));
				this.$msgModal.modal("show");
				return false;
			} else {
				var self = this;
				this.$processModal.show();
				setTimeout(function(){
					var stats = new $.Stats();
					var nn = parseInt(2*stats.$R);
					var E = null;
					var s1 = stats.simpleEM(scores,100,100,self);
					var E = stats.computeExpScores(s1,nn,self);
					var b = stats.getMaxAndArgmax(E);
					var max = b[0].toFixed(2).toString();
					var imax = b[1];
					var jmax = b[2];
					var aimStr = stats.getRegion(imax,jmax,nn,nn);
					var N = scores.length;

					self.$statsNotes.text(n);
					self.$maxExpectedScore.text(max);
					self.$aimFor.text(aimStr);
					self.$sigma.text(Math.sqrt(s1).toFixed(2));

					self._heatmapClear();
					self._heatmapRepaint(E,imax,jmax);

					self.$processModal.hide();
				},1000);

			}
		},

		boardToggle: function() {
			var self = this.$showBoardBtn;			
			if(self.prop("checked")){
				this.$ovCan.show();
			}else{
				this.$ovCan.hide();
			}
		},

		optToggle: function() {
			var self = this.$showOptBtn;			
			if(self.prop("checked")){
				$('#optimumDot').show();
			}else{
				$('#optimumDot').hide();
			}
		},

		changeColor: function() {
			
		    this._getCssValuePrefix(this.$legend,this.$colorSet);

			this.$msgModal.find('.modal-title').text("Change Color Scheme");
			this.$msgModal.find('.modal-body').text("Test Text");
			this.$msgModal.modal("show");

		},

		_getCssValuePrefix: function(target,colors) {
			var numStops = colors.length - 1;
			var percentage = Math.ceil(10 / numStops);
			var l=document.getElementById("legend");
			var ctx=l.getContext("2d");

			var grd=ctx.createLinearGradient(0,0,l.width,l.height);
			for(var i=0; i < colors.length; i++) {
				grd.addColorStop("0."+i,colors[i]._hex);
			}
			ctx.fillStyle=grd;
			ctx.fillRect(0,0,l.width,l.height);
			console.log(numStops);
			console.log(percentage);
			
		},

		_checkNumbers: function(scores) {
			var invalids = [];
			var atCapacity = false;

			for (var i=0; i<scores.length; i++) {
				var s = scores[i];
				if (((0 <= s) && (s <= 20)) || s == 21 || s == 22 || s == 24 || s == 25 || s == 26 ||
					s == 27 || s == 28 || s == 30 || s == 32 || s == 33 || s == 34 || s == 36 || 
					s == 38 || s == 39 || s == 40 || s == 42 || s == 45 || s == 48 || s == 50 ||
					s == 51 || s == 54 || s == 57 || s == 60) {
					continue;
			    } else {		
					if (invalids.length == 10) {
					    atCapacity = true;
					    break;
					} else {
						invalids.push(s);
					}
			    }
			}

			return invalids;
		},

		_heatmapClear: function() {
			this.$hmCtx.clearRect(0,0,this.$heatmapBox.width(),this.$heatmapBox.height());
		},

		_heatmapRepaint: function(E,imax,jmax) {
			this._setColorScheme(this.$colorSet);
			
			var w = this.$hmCanvas.width;
			var h = this.$hmCanvas.height;
			var n = E.length;
			var numIter = E.length/100;
			var min = E[0][0];
		    var max = E[0][0];
		    for (var i=0; i<n; i++) {
				for (var j=0; j<n; j++) {
			    	if (E[i][j] < min) min = E[i][j];
			    	if (E[i][j] > max) max = E[i][j];
				}
		    }

		    for (var i=0; i<n; i++) {
				for (var j=0; j<n; j++) {
			    	this.$hmCtx.fillStyle = this._getHeatColor(E[i][j],min,max);
			    	this.$hmCtx.fillRect(i,n-1-j,1,1);
				}
		    }

		    var optX = ((imax-1)/w)*100;
		    var optY = ((h-3-jmax)/h)*100;

		    this.$heatmapBox.find('#heatMapImage').remove();
		    var image = new Image();
		    image.src = this.$hmCanvas.toDataURL("image/png");
		    image.className = "img-responsive";
		    image.id = "heatMapImage";

		    this.$heatmapBox.append(image);
		    this.$heatmapBox.find('#optimumDot').css({'top':optY+'%','left':optX+'%'});
		    $("#heatMapImage").css({'width':'100%','height':'99.5%','position':'absolute','top':'0px','left':'1px','z-index':'0'});
		},

		_getHeatColor: function(t,min,max) {
			var i = parseInt(Math.round((t-min)/(max-min)*(this.$numColors-1)));
		    return this.$heatColors[i]["_hex"];
		},

		_setColorScheme: function(colors) {
			var k = colors.length;
	    	var r = [];
	    	var g = [];
	    	var b = [];

	    	for (var j=0; j<k; j++) {
				r[j] = colors[j]._red;
				g[j] = colors[j]._green;
				b[j] = colors[j]._blue;
		    }

	    	this.$colorScheme = colors;
	    	this.$numColors = Math.max(k*20,50);
	    	var m = parseFloat(this.$numColors-1);   
	    	var mm = parseFloat(k-1);

	    	this.$heatColors = [this.$numColors];
	    	for (var i=0; i<this.$numColors; i++) {
				var x = parseFloat(i/m*mm);
				var j = parseInt(x);
				if (j <= k-2) {
		    		this.$heatColors[i] = new Color({red:r[j]*(1-x+j) + r[j+1]*(x-j),green:g[j]*(1-x+j) + g[j+1]*(x-j),blue:b[j]*(1-x+j) + b[j+1]*(x-j)});
				} else {
		    		this.$heatColors[i] = new Color({red:r[j],green:g[j],blue:b[j]});
				}
	    	}
		}
	}
})(jQuery);

// STATS CLASS
(function ($){
	"use strict"

	$.Stats = function() {
		this.init();
	}

	$.Stats.prototype = {
		init: function() {
			// board dimensions in mm (measured from center of board)
			this.$R1 = 6.35; 	// distance to double bull
			this.$R2 = 15.9; 	// distance to singel bull
			this.$R3 = 99; 		// distance to inside triple ring
			this.$R4 = 107;		// distance to outside triple ring
			this.$R5 = 162;		// distance to inside double ring
			this.$R = 170;		// distance to outside double ring
			this.$d = [20,1,18,4,13,6,10,15,2,17,3,19,7,16,8,11,14,9,12,5];
			this.$ii = [2,9,11,4,20,6,13,15,18,7,16,19,5,17,8,14,10,3,12,1];
			this.$pow2 = [1,2,4,8,16,32,64,128,256,512,1024];
		},

		simpleEM: function(scores,sInit,numIter,da) {
			var s = parseInt(sInit);
			var inc = numIter/10;
			for (var i=0; i<numIter; i++) {
				s = this._simpleStep(scores,s);
			}
			return s;
		},

		computeExpScores: function(s,n,da) {
			// round up to the nearest power of 2
			var m = 1;
			for (var i=0; i<this.$pow2.length; i++) {
				m = this.$pow2[i];
				if( n <= this.$pow2[i]) {
					break;
				}
			}

			// number of millimeters per pixel
			var c = 2*this.$R/n;

			// build the scores array
			var S = this._createArray(2*m+1,2*m+1);
			for (var i=0; i<=2*m; i++) {
				for (var j=1; j<=2*m; j++) {
					S[i][j] = this._score((i-m)*c,(j-m)*c);
				}
			}

			// helpful constants
			var n1 = Math.floor(n/2.0);
			var n2 = Math.ceil(n/2.0);

			// if the variance is 0, just return the scores array!
			if (s == 0) {
				var E = this._createArray(n,n);
				for (var i=m-n1+1; i<=m+n2; i++) {
					for (var j=m-n1+1; j<=m+n2; j++) {
						E[i-m+n1-1][j-m+n1-1] = S[i][j];
					}
				}
				return E;
			}

			// build the Gaussian density arrays
			var g1 = this._createArray(2*m+1);
			var g2 = this._createArray(2*m+1);
			for (var i=1; i<=2*m; i++) {
				g1[i] = Math.exp(-(i-m)*(i-m)*c*c/(2*s))/Math.sqrt(2*Math.PI*s)*c;
				g2[i] = g1[i];
			}

			// compute the FT of g1 and g2
			var g1f = this._createArray(4*m+1);
			var g2f = this._createArray(4*m+1);
			this._twofft(g1,g2,g1f,g2f,2*m);

			// compute the FT of each row of S
			var A = this._createArray(2*m+1,4*m+1);
			for (var i=1; i<=2*m-1; i+=2) {
				this._twofftrow(S,i,i+1,A,2*m);
			}

			// multiply every row of A by g2f
			var re,im;
			for (var i=1; i<=2*m; i+=1) {
				for (var j=1; j<=4*m-1; j+=2) {
					re = A[i][j]*g2f[j] - A[i][j+1]*g2f[j+1];
					im = A[i][j]*g2f[j+1] + A[i][j+1]*g2f[j];
					A[i][j] = re;
					A[i][j+1] = im;
				}
			}

			// compute the inverse FT of every row of A
			for (var i=1; i<=2*m; i++) {
		    	this._four1row(A,i,2*m,-1);
			}

			// take the real part of each row, and switch around some columns
			var AA = this._createArray(2*m+1,2*m+1);
			for (var i=1; i<=2*m; i++) {
			    for (var j=1; j<=m; j++) {
					AA[i][j+m] = A[i][2*j-1]/(2*m);
			    }
			    for (var j=m+1; j<=2*m; j++) {
					AA[i][j-m] = A[i][2*j-1]/(2*m);
			    }
			}

			// compute the FT of every column of AA
			var AAA = this._createArray(4*m+1,2*m+1);
			for (var j=1; j<=2*m-1; j+=2) {
			    this._twofftcol(AA,j,j+1,AAA,2*m);
			}

			// multiply every column of AAA by g1f
			for (var j=1; j<=2*m; j++) {
			    for (var i=1; i<=4*m-1; i+=2) {
					re = AAA[i][j]*g1f[i] - AAA[i+1][j]*g1f[i+1];
					im = AAA[i][j]*g1f[i+1] + AAA[i+1][j]*g1f[i];
					AAA[i][j] = re;
					AAA[i+1][j] = im;
			    }
			}

			// compute the inverse FT of every column of AAA
			for (var j=1; j<=2*m; j++) {
			    this._four1col(AAA,j,2*m,-1);
			}

			// take the real part of each column, switch around some rows,
			// and strip away some part of the array on each side
			var E = this._createArray(n,n);
			for (var i=1; i<=n1; i++) {
			    for (var j=m-n1+1; j<=m+n2; j++) {
					E[i+n2-1][j-m+n1-1] = AAA[2*i-1][j]/(2*m);
			    }
			}
			for (var i=2*m-n2+1; i<=2*m; i++) {
			    for (var j=m-n1+1; j<=m+n2; j++) {
					E[i-2*m+n2-1][j-m+n1-1] = AAA[2*i-1][j]/(2*m);
			    }
			}

			// help the gc out
			S=null; g1=null; g2=null; g1f=null; g2f=null; 
			A=null; AA=null; AAA=null;

			return E;
		},

		generalEM: function(scores,s1Init,s2Init,s3Init,numIter,da) {
			
			var s1 = parseInt(s1Init); 
			var s2 = parseInt(s2Init); 
			var s3 = parseInt(s3Init);

			var inc = numIter/10;

			for (var i=0; i<numIter; i++) {
			    var A = this._generalStep(scores, s1, s2, s3);
			    s1 = A[0];
			    s2 = A[1];
			    s3 = A[2];	    
			    if (i % inc == 0) {
					da._updateProgressVal(50*i/numIter);
			    }
			}

			return [s1, s2, s3];
		},

		computeExpScoresGeneral: function(s1,s2,s3,n,da) {
			// round up to the nearest power of 2					     
			var m = 1;
			for (var i=0; i<this.$pow2.length; i++) {
			    m = this.$pow2[i];
			    if (n <= this.$pow2[i]) {
					break;
			    }
			}

			// note: due to the design of all the FFT code (taken from Numerical 
			// Recipes in C), all of our arrays must be unit-based!

			// another note: all of our matrices here are complex. the layout is that
			// there are twice as many columns as a corresponding real array representing
			// the same dimension, so that m[i][j] and m[i][j+1] represent corresponding
			// real and complex parts of the same number

			// number of millimeters per pixel
			var c = 2*this.$R/n;
		
			// build the scores array
			S = this._createArray(2*m+1,4*m+1);
			for (var i=1; i<=2*m; i++) {
			    for (var j=1; j<=2*m; j++) {
					S[i][2*j-1] = this._score((i-m)*c,(j-m)*c);
					S[i][2*j] = 0;
			    }
			}
			da._updateProgressVal(50+1/7*50);

	       	// helpful constants
			var n1 = parseInt(Math.floor(n/2.0));
			var n2 = parseInt(Math.ceil(n/2.0));

			// build the Gaussian density array
			var G = this._createArray(2*m+1,4*m+1);
			var det = s1*s2-s3*s3;
			for (var i=1; i<=2*m; i++) {
			    for (var j=1; j<=2*m; j++) {
				G[i][2*j-1] = Math.exp(-(s2*(i-m)*(i-m) - 2*s3*(i-m)*(j-m) + s1*(j-m)*(j-m))*c*c/(2*det)) /(2*Math.PI*Math.sqrt(det))*c*c;
				G[i][2*j] = 0;
			    }
			}
			da._updateProgressVal(50+2/7*50);

			// compute the FT of each matrix (in-place)
			this._four2(S,2*m,1);
			da._updateProgressVal(50+3/7*50);

			this._four2(G,2*m,1);
			da._updateProgressVal(50+4/7*50);

			// multiply S and G together (store the result in S)
			var re,im;
			for (var i=1; i<=2*m; i+=1) {
			    for (var j=1; j<=4*m-1; j+=2) {
					re = S[i][j]*G[i][j] - S[i][j+1]*G[i][j+1];
					im = S[i][j]*G[i][j+1] + S[i][j+1]*G[i][j];
					S[i][j] = re;
					S[i][j+1] = im;
			    }
			}
			da._updateProgressVal(50+5/7*50);

			// compute the inverse FT of S (in-place)
			this._four2(S,2*m,-1);
			da._updateProgressVal(50+6/7*50);

			// switch around some rows and some columns, take the real part
			var EE = this._createArray(2*m+1,2*m+1);
			for (var i=1; i<=m; i++) {
			    for (var j=1; j<=m; j++) {
					EE[i+m][j+m] = S[i][2*j-1]/(4*m*m);
			    }
			    for (var j=m+1; j<=2*m; j++) {
					EE[i+m][j-m] = S[i][2*j-1]/(4*m*m);
			    }
			}
			for (var i=m+1; i<=2*m; i++) {
			    for (var j=1; j<=m; j++) {
					EE[i-m][j+m] = S[i][2*j-1]/(4*m*m);
			    }
			    for (var j=m+1; j<=2*m; j++) {
					EE[i-m][j-m] = S[i][2*j-1]/(4*m*m);
			    }
			}

			// trim away on each side
			var E = this._createArray(n,n);
			for (var i=m-n1; i<=m+n2-1; i++) {
			    for (var j=m-n1; j<=m+n2-1; j++) {
					E[i-m+n1][j-m+n1] = EE[i][j];
			    }
			}
			da._updateProgressVal(100);

			// help the gc out
			S=null; G=null; EE=null;

			return E;
		},

		getMaxAndArgmax: function(data) {
			var max = data[0][0];
			var ii=0,jj=0;

			for(var i=0; i<data.length; i++) {
				for(var j=0; j<data.length; j++) {
					if(data[i][j] > max) {
						max = data[i][j];
						ii = i;
						jj = j;
					}
				}
			}

			return [max,ii,jj];
		},

		getRegion: function(i,j,n1,n2) {
			return this._getRegion((i+1-n1/2)*2*this.$R/n1, (j+2-n2/2)*2*this.$R/n2);
		},

		_createArray: function(length) {
		    var arr = new Array(length || 0),
		        i = length;

		    if (arguments.length > 1) {
		        var args = Array.prototype.slice.call(arguments, 1);
		        while(i--) arr[length-1 - i] = this._createArray.apply(this, args);
		    }

		    return arr;
		},

		_simpleStep: function(scores,s) {
			var a = this._createArray(6);
			var b = this._createArray(6);

			// calculate the integral (pg.6 in paper) over each region
			// note: the regions are defined as follows
			// 0 - double bullseye
			// 1 - single bullseye
			// 2 - single 
			// 3 - double ring
			// 4 - triple ring 
			// 5 - off the board
			a[0] = this._integNumerator(s,0,this.$R1);
			a[1] = this._integNumerator(s,this.$R1,this.$R2);
			a[2] = this._integNumerator(s,this.$R2,this.$R3)/20 + this._integNumerator(s,this.$R4,this.$R5)/20;
			a[3] = this._integNumerator(s,this.$R5,this.$R)/20;
			a[4] = this._integNumerator(s,this.$R3,this.$R4)/20;
			a[5] = this._integNumerator(s,this.$R,-1);
		
			// calculate the integral over each region
			b[0] = this._integDenominator(s,0,this.$R1); 
			b[1] = this._integDenominator(s,this.$R1,this.$R2);
			b[2] = this._integDenominator(s,this.$R2,this.$R3)/20 + this._integDenominator(s,this.$R4,this.$R5)/20;
			b[3] = this._integDenominator(s,this.$R5,this.$R)/20;
			b[4] = this._integDenominator(s,this.$R3,this.$R4)/20;
			b[5] = this._integDenominator(s,this.$R,-1);

			var n = scores.length;
			var e = 0;
			for(var i=0; i<n; i++) {
				e += this._computeExp(scores[i],a,b);

			}
			
			return e/(2*n);
		},

		_computeExp: function(x,a,b) {
			// return the appropriate expectation, based on how the score can be achieved
			if (x==1 || x==5 || x==7 || x==11 || x==13 || x==17 || x==19) {
			    // these numbers can only be achieved by hitting a single
			    return a[2]/b[2];
			} else if (x==2 || x==4 || x==8 || x==10 || x==14 || x==16 || x==20) {
			    // single or double
			    return (a[2]+a[3])/(b[2]+b[3]);
			} else if (x==3 || x==9 || x==15) {
			    // single or triple region
			    return (a[2]+a[4])/(b[2]+b[4]);
			} else if (x==6 || x==12 || x==18) {
			    // single, double, or triple
			    return (a[2]+a[3]+a[4])/(b[2]+b[3]+b[4]);
			} else if (x==24 || x==30 || x==36) {
			    // double or triple
			    return (a[3]+a[4])/(b[3]+b[4]);
			} else if (x==22 || x==26 || x==28 || x==32 || x==34 || x==38 || x==40) {
			    // double only
			    return a[3]/b[3];
			} else if (x==21 || x==27 || x==33 || x==39 || x==42 || x==45 || x==48 || 
				 x==51 || x==54 || x==57 || x==60) {
			    // triple only
			    return a[4]/b[4];
			} else if (x==25) {
			    // single bullseye
			    return a[1]/b[1];
			} else if (x==50) {
			    // double bullseye
			    return a[0]/b[0];
			} else {
			    // outside the board
			    return a[5]/b[5];
			}
		},

		_integNumerator: function(s,r1,r2) {
			// r2 is assumed to be infinity
			if (r2 == -1) {
			    return (r1*r1+2*s)*Math.exp(-r1*r1/(2*s));
			} else { 
			    return (r1*r1+2*s)*Math.exp(-r1*r1/(2*s)) - (r2*r2+2*s)*Math.exp(-r2*r2/(2*s));
			}
		},

		_integDenominator: function(s,r1,r2) {
			// r2 is assumed to be infinity
			if (r2 == -1) {
			    return Math.exp(-r1*r1/(2*s));
			} else {
			    return Math.exp(-r1*r1/(2*s)) - Math.exp(-r2*r2/(2*s));
			}
		},

		_generalStep: function(scores,s1,s2,s3) {
			var n = scores.length;
			var A = this._createArray(3);
			A[0] = 0; 
			A[1] = 0;
			A[2] = 0;
			
			for (var i=0; i<n; i++) {
			    var B = this._simulateExp(scores[i], s1, s2, s3);
			    A[0] += B[0];
			    A[1] += B[1];
			    A[2] += B[2];
			}

			return [A[0]/n, A[1]/n, A[2]/n];
		},

		_simulateExp: function(x,s1,s2,s3) {
			var numImpSamples = 1000;
			var det = s1*s2 - s3*s3;
			var A = this._createArray(3);
			A[0] = 0;
			A[1] = 0;
			A[2] = 0;
			var W = 0;
			
			for (var i=0; i<numImpSamples; i++) {
			    var z = this._randomPt(x);
			    var w = Math.exp(-(s2*z[0]*z[0] - 2*s3*z[0]*z[1] + 
						  s1*z[1]*z[1])/(2*det));
			    A[0] += w*z[0]*z[0];
			    A[1] += w*z[1]*z[1];
			    A[2] += w*z[0]*z[1];
			    W += w;
			}
			
			A[0] /= W;
			A[1] /= W;
			A[2] /= W;
			return A;
		},

		_randomPt: function(x) {
			var u = Math.seedrandom(1);
			var ar1 = Math.PI*this.$R1*this.$R1;              // double bullseye
		    var ar2 = Math.PI*(this.$R2*this.$R2 - this.$R1*this.$R1);    // single bullseye
		    var ar3 = Math.PI*(this.$R3*this.$R3 - this.$R2*this.$R2)/20; // lower single
		    var ar4 = Math.PI*(this.$R5*this.$R5 - this.$R4*this.$R4)/20; // upper single
		    var ar5 = Math.PI*(this.$R*this.$R - this.$R5*this.$R5)/20;   // double
		    var ar6 = Math.PI*(this.$R4*this.$R4 - this.$R3*this.$R3)/20; // triple

			if (x==1 || x==5 || x==7 || x==11 || x==13 || x==17 || x==19) {
			    // single only
			    if (u <= ar3/(ar3+ar4)) {
					return this._randomSlicePt(x,this.$R2,this.$R3);
			    } else {
					return this._randomSlicePt(x,this.$R4,this.$R5);
			    }
			} else if (x==2 || x==4 || x==8 || x==10 || x==14 || x==16 || x==20) {
			    // single or double
			    if (u <= ar5/(ar5+ar3+ar4)) {
					return this._randomSlicePt(x/2,this.$,R);
			    } else if (u <= (ar5+ar3)/(ar5+ar3+ar4)) {
					return this._randomSlicePt(x,this.$R2,this.$R3);
			    } else {
					return this._randomSlicePt(x,this.$R4,this.$R5);
			    }
			} else if (x==3 || x==9 || x==15) {
			    // single or triple
			    if (u <= ar6/(ar6+ar3+ar4)) {
					return this._randomSlicePt(x/3,this.$R3,this.$R4);
			    } else if (u <= (ar6+ar3)/(ar6+ar3+ar4)) {
					return this._randomSlicePt(x,this.$R2,this.$R3);
			    } else {
					return this._randomSlicePt(x,this.$R4,this.$R5);
			    }
			} else if (x==6 || x==12 || x==18) {
			    // single, double, or triple
			    if (u <= ar6/(ar6+ar5+ar3+ar4)) {
					return this._randomSlicePt(x/3,this.$R3,this.$R4);
			    } else if (u <= (ar6+ar5)/ (ar6+ar5+ar3+ar4)) {
					return this._randomSlicePt(x/2,this.$R5,this.$R); 
			    } else if (u <= (ar6+ar5+ar3)/ (ar6+ar5+ar3+ar4)) {
					return this._randomSlicePt(x,this.$R2,this.$R3);
			    } else {
					return this._randomSlicePt(x,this.$R4,this.$R5); 
			    } 
			} else if (x==24 || x==30 || x==36) {
			    // double or triple
			    if (u <= ar6/(ar6+ar5)) {
					return this._randomSlicePt(x/3,this.$R3,this.$R4);
			    } else {
					return this._randomSlicePt(x/2,this.$R5,this.$R);
			    }
			} else if (x==22 || x==26 || x==28 || x==32 || x==34 || x==38 || x==40) {
			    // double only
			    return this._randomSlicePt(x/2,this.$R5,this.$R);
			} else if (x==21 || x==27 || x==33 || x==39 || x==42 || x==45 || x==48 || x==51 || x==54 || x==57 || x==60) {
			    // triple only
			    return this._randomSlicePt(x/3,this.$R3,this.$R4);
			} else if (x==25) {
			    // single bullseye
			    return this._randomCirclePt(this.$R1,this.$R2);
			} else if (x==50) {
			    // double bullseye
			    return this._randomCirclePt(0,this.$R1);
			} else {
			    // outside the board
			    // a bit of a cheat, the second number should be infinity
			    return this._randomCirclePt(this.$R,10*this.$R);
			}
		},

		_randomSlicePt: function(x,r1,r2) {
			var k = this.$ii[x-1];
			var u = Math.seedrandom(2);
			var th = -2*Math.PI/40 + (k-1)*2*Math.PI/20 + 2*Math.PI/20*u;
			th = Math.PI/2 - th;
			var r = this._randomR(r1,r2);
			return [r*Math.cos(th), r*Math.sin(th)];
		},

		_randomCirclePt: function(r1,r2) {
			var u = Math.seedrandom(3);
			var th = 2*Math.PI*u;
			var r = this._randomR(r1,r2);
			return [r*Math.cos(th), r*Math.sin(th)];
		},

		_randomR: function(r1,r2) {
			var u = Math.seedrandom(4);
			return Math.sqrt(r1*r1 + (r2*r2-r1*r1)*u);
		},

		_twofft: function(data1,data2,fft1,fft2,n) {
			var nn3,nn2,jj,j,rep,rem,aip,aim;

			nn3=1+(nn2=2+n+n);
			for (j=1,jj=2;j<=n;j++,jj+=2) { 
			    fft1[jj-1]=data1[j]; 
			    fft1[jj]=data2[j];
			}
			this._four1(fft1,n,1);
			fft2[1]=fft1[2];
			fft1[2]=fft2[2]=0.0;
			for (j=3;j<=n+1;j+=2) {
			    rep=0.5*(fft1[j]+fft1[nn2-j]); 
			    rem=0.5*(fft1[j]-fft1[nn2-j]); 
			    aip=0.5*(fft1[j+1]+fft1[nn3-j]);
			    aim=0.5*(fft1[j+1]-fft1[nn3-j]);
			    fft1[j]=rep; 
			    fft1[j+1]=aim;
			    fft1[nn2-j]=rep;
			    fft1[nn3-j] = -aim;
			    fft2[j]=aip;
			    fft2[j+1] = -rem;
			    fft2[nn2-j]=aip;
			    fft2[nn3-j]=rem;
			}
		},

		_twofftrow: function(data,k1,k2,fft,n) {
			var nn3,nn2,jj,j,rep,rem,aip,aim;

			nn3=1+(nn2=2+n+n);
			for (j=1,jj=2;j<=n;j++,jj+=2) { 
			    fft[k1][jj-1]=data[k1][j]; 
			    fft[k1][jj]=data[k2][j];
			}
			this._four1row(fft,k1,n,1);
			fft[k2][1]=fft[k1][2];
			fft[k1][2]=fft[k2][2]=0.0;
			for (j=3;j<=n+1;j+=2) {
			    rep=0.5*(fft[k1][j]+fft[k1][nn2-j]); 
			    rem=0.5*(fft[k1][j]-fft[k1][nn2-j]); 
			    aip=0.5*(fft[k1][j+1]+fft[k1][nn3-j]);
			    aim=0.5*(fft[k1][j+1]-fft[k1][nn3-j]);
			    fft[k1][j]=rep; 
			    fft[k1][j+1]=aim;
			    fft[k1][nn2-j]=rep;
			    fft[k1][nn3-j] = -aim;
			    fft[k2][j]=aip;
			    fft[k2][j+1] = -rem;
			    fft[k2][nn2-j]=aip;
			    fft[k2][nn3-j]=rem;
			}
		},

		_twofftcol: function(data,k1,k2,fft,n) {
			var nn3,nn2,jj,j,rep,rem,aip,aim;

			nn3=1+(nn2=2+n+n);
			for (j=1,jj=2;j<=n;j++,jj+=2) { 
			    fft[jj-1][k1]=data[j][k1]; 
			    fft[jj][k1]=data[j][k2];
			}
			this._four1col(fft,k1,n,1);
			fft[1][k2]=fft[2][k1];
			fft[2][k1]=fft[2][k2]=0.0;
			for (j=3;j<=n+1;j+=2) {
			    rep=0.5*(fft[j][k1]+fft[nn2-j][k1]); 
			    rem=0.5*(fft[j][k1]-fft[nn2-j][k1]); 
			    aip=0.5*(fft[j+1][k1]+fft[nn3-j][k1]);
			    aim=0.5*(fft[j+1][k1]-fft[nn3-j][k1]);
			    fft[j][k1]=rep; 
			    fft[j+1][k1]=aim;
			    fft[nn2-j][k1]=rep;
			    fft[nn3-j][k1] = -aim;
			    fft[j][k2]=aip;
			    fft[j+1][k2] = -rem;
			    fft[nn2-j][k2]=aip;
			    fft[nn3-j][k2]=rem;
			}
		},

		_four1row: function(data,k,nn,isign) {
			var n,mmax,m,j,istep,i,wtemp,wr,wpr,wpi,wi,theta,temp,tempr,tempi;

			n=nn << 1;
			j=1;
			for (i=1;i<n;i+=2) { 
			    if (j > i) { 
				// SWAP(data[k][j],data[k][i]); 
				temp=data[k][j];
				data[k][j]=data[k][i];
				data[k][i]=temp;
				// SWAP(data[k][j+1],data[k][i+1]);
				temp=data[k][j+1];
				data[k][j+1]=data[k][i+1];
				data[k][i+1]=temp;
			    }
			    m=n >>> 1;
			    while (m >= 2 && j > m) {
					j -= m;
					m=m >>> 1;
			    }
			    j += m;
			}
			mmax=2;
			while (n > mmax) { 
			    istep=mmax << 1;
			    theta=isign*(6.28318530717959/mmax); 
			    wtemp=Math.sin(0.5*theta);
			    wpr = -2.0*wtemp*wtemp;
			    wpi=Math.sin(theta);
			    wr=1.0;
			    wi=0.0;
			    for (m=1;m<mmax;m+=2) { 
					for (i=m; i<=n;i+=istep) {
					    j=i+mmax;
					    tempr=wr*data[k][j]-wi*data[k][j+1]; 
					    tempi=wr*data[k][j+1]+wi*data[k][j];
					    data[k][j]=data[k][i]-tempr;
					    data[k][j+1]=data[k][i+1]-tempi;
					    data[k][i] += tempr;
					    data[k][i+1] += tempi;
					}
					wr=(wtemp=wr)*wpr-wi*wpi+wr; 
					wi=wi*wpr+wtemp*wpi+wi;
			    }
			    mmax=istep;
			}
		},

		_four1col: function(data,k,nn,isign) {
			var n,mmax,m,j,istep,i,wtemp,wr,wpr,wpi,wi,theta,temp,tempr,tempi;

			n=nn << 1;
			j=1;
			for (i=1;i<n;i+=2) { 
			    if (j > i) { 
					// SWAP(data[j][k],data[i][k]); 
					temp=data[j][k];
					data[j][k]=data[i][k];
					data[i][k]=temp;
					// SWAP(data[j+1][k],data[i+1][k]);
					temp=data[j+1][k];
					data[j+1][k]=data[i+1][k];
					data[i+1][k]=temp;
			    }
			    m=n >>> 1;
			    while (m >= 2 && j > m) {
					j -= m;
					m=m >>> 1;
			    }
			    j += m;
			}
			mmax=2;
			while (n > mmax) { 
			    istep=mmax << 1;
			    theta=isign*(6.28318530717959/mmax); 
			    wtemp=Math.sin(0.5*theta);
			    wpr = -2.0*wtemp*wtemp;
			    wpi=Math.sin(theta);
			    wr=1.0;
			    wi=0.0;
			    for (m=1;m<mmax;m+=2) { 
					for (i=m; i<=n;i+=istep) {
					    j=i+mmax;
					    tempr=wr*data[j][k]-wi*data[j+1][k]; 
					    tempi=wr*data[j+1][k]+wi*data[j][k];
					    data[j][k]=data[i][k]-tempr;
					    data[j+1][k]=data[i+1][k]-tempi;
					    data[i][k] += tempr;
					    data[i+1][k] += tempi;
					}
					wr=(wtemp=wr)*wpr-wi*wpi+wr; 
					wi=wi*wpr+wtemp*wpi+wi;
			    }
			    mmax=istep;
			}
		},

		_four1: function(data,nn,isign) {
			var n,mmax,m,j,istep,i,wtemp,wr,wpr,wpi,wi,theta,temp,tempr,tempi;

			n=nn << 1;
			j=1;
			for (i=1;i<n;i+=2) { 
			    if (j > i) { 
					// SWAP(data[j],data[i]); 
					temp=data[j];
					data[j]=data[i];
					data[i]=temp;
					// SWAP(data[j+1],data[i+1]);
					temp=data[j+1];
					data[j+1]=data[i+1];
					data[i+1]=temp;
			    }
			    m=n >>> 1;
			    while (m >= 2 && j > m) {
					j -= m;
					m=m >>> 1;
			    }
			    j += m;
			}
			mmax=2;
			while (n > mmax) { 
			    istep=mmax << 1;
			    theta=isign*(6.28318530717959/mmax); 
			    wtemp=Math.sin(0.5*theta);
			    wpr = -2.0*wtemp*wtemp;
			    wpi=Math.sin(theta);
			    wr=1.0;
			    wi=0.0;
			    for (m=1;m<mmax;m+=2) { 
					for (i=m; i<=n;i+=istep) {
					    j=i+mmax;
					    tempr=wr*data[j]-wi*data[j+1]; 
					    tempi=wr*data[j+1]+wi*data[j];
					    data[j]=data[i]-tempr;
					    data[j+1]=data[i+1]-tempi;
					    data[i] += tempr;
					    data[i+1] += tempi;
					}
					wr=(wtemp=wr)*wpr-wi*wpi+wr; 
					wi=wi*wpr+wtemp*wpi+wi;
			    }
			    mmax=istep;
			}
		},

		_four2: function(data,nn,isign) {
			// compute the 1D FT of every row
			for (var i=1; i<=nn; i++) {
			    this._four1row(data,i,nn,isign);
			}

			// compute the 1D FT of every column
			var a = this._createArray(2*nn+1);
			for (var j=1; j<=2*nn-1; j+=2) {
			    // copy into the array a
			    for (var i=1; i<=nn; i++) {
					a[2*i-1] = data[i][j];
					a[2*i] = data[i][j+1];
			    }
			    // compute the 1D FT of a
			    this._four1(a,nn,isign);
			    // copy back into the matrix data
			    for (var i=1; i<=nn; i++) {
					data[i][j] = a[2*i-1];
					data[i][j+1] = a[2*i];
			    }
			}
		},

		_score: function(x,y) {
			// compute the radius
			var r = Math.sqrt(x*x + y*y);
		
			// check if it's off the board (do this for speed)
			if (r > this.$R) return 0;
		
			// check for a double bullseye
			if (r <= this.$R1) return 50;
		
			// check for a single bullseye
			if (r <= this.$R2) return 25;
		
			// get the angle
			var th = Math.atan2(y, x);
			var phi = (Math.PI/2+Math.PI/20-th) % (2*Math.PI);
			if (phi < 0) phi += 2*Math.PI;
		
			// now get the number
			var i = parseInt(phi/(2*Math.PI)*20);
			if (i < 0) i = 0;
			if (i >= 19) i = 19;
			var n = this.$d[i];
		
			// check for a single
			if (r <= this.$R3) return n;
		
			// check for a triple
			if (r <= this.$R4) return 3*n;
		
			// check for a single
			if (r <= this.$R5) return n;
		
			//f we got here, it must be a double
			return 2*n;
		},

		_getRegion: function(x,y) {
			// compute the radius
			var r = Math.sqrt(x*x + y*y);
		
			// check if it's off the board (do this for speed)
			if (r > this.$R) return "Off";
		
			// check for a double bullseye
			if (r <= this.$R1) return "DB";
		
			// check for a single bullseye
			if (r <= this.$R2) return "SB";
		
			// get the angle
			var th = Math.atan2(y, x);
			var phi = (Math.PI/2+Math.PI/20-th) % (2*Math.PI);
			if (phi < 0) phi += 2*Math.PI;
		
			// now get the number
			var i = parseInt(phi/(2*Math.PI)*20);
			if (i < 0) i = 0;
			if (i >= 19) i = 19;
			var n = this.$d[i];
		
			// check for a single
			if (r <= this.$R3) return "S"+n;
		
			// check for a triple=
			if (r <= this.$R4) return "T"+n;
		
			// check for a single
			if (r <= this.$R5) return "S"+n;
		
			//f we got here, it must be a double
			return "D"+n;
		}
	}
})(jQuery);

var Color = (function(window){

	var Events = {
		RGB_UPDATED : 'RGBUpdated',
		HSL_UPDATED : 'HSLUpdated',
		HSV_UPDATED : 'HSVUpdated',
		HEX_UPDATED : 'HexUpdated',
		INT_UPDATED : 'IntUpdated',
		UPDATED : 'updated'
	};

	var namedColors = {
		'transparent':'rgba(0, 0, 0, 0)','aliceblue':'#F0F8FF','antiquewhite':'#FAEBD7','aqua':'#00FFFF','aquamarine':'#7FFFD4',
		'azure':'#F0FFFF','beige':'#F5F5DC','bisque':'#FFE4C4','black':'#000000','blanchedalmond':'#FFEBCD','blue':'#0000FF','blueviolet':'#8A2BE2',
		'brown':'#A52A2A','burlywood':'#DEB887','cadetblue':'#5F9EA0','chartreuse':'#7FFF00','chocolate':'#D2691E','coral':'#FF7F50',
		'cornflowerblue':'#6495ED','cornsilk':'#FFF8DC','crimson':'#DC143C','cyan':'#00FFFF','darkblue':'#00008B','darkcyan':'#008B8B','darkgoldenrod':'#B8860B',
		'darkgray':'#A9A9A9','darkgrey':'#A9A9A9','darkgreen':'#006400','darkkhaki':'#BDB76B','darkmagenta':'#8B008B','darkolivegreen':'#556B2F',
		'darkorange':'#FF8C00','darkorchid':'#9932CC','darkred':'#8B0000','darksalmon':'#E9967A','darkseagreen':'#8FBC8F','darkslateblue':'#483D8B',
		'darkslategray':'#2F4F4F','darkslategrey':'#2F4F4F','darkturquoise':'#00CED1','darkviolet':'#9400D3','deeppink':'#FF1493','deepskyblue':'#00BFFF',
		'dimgray':'#696969','dimgrey':'#696969','dodgerblue':'#1E90FF','firebrick':'#B22222','floralwhite':'#FFFAF0','forestgreen':'#228B22',
		'fuchsia':'#FF00FF','gainsboro':'#DCDCDC','ghostwhite':'#F8F8FF','gold':'#FFD700','goldenrod':'#DAA520','gray':'#808080','grey':'#808080',
		'green':'#008000','greenyellow':'#ADFF2F','honeydew':'#F0FFF0','hotpink':'#FF69B4','indianred':'#CD5C5C','indigo':'#4B0082','ivory':'#FFFFF0',
		'khaki':'#F0E68C','lavender':'#E6E6FA','lavenderblush':'#FFF0F5','lawngreen':'#7CFC00','lemonchiffon':'#FFFACD','lightblue':'#ADD8E6',
		'lightcoral':'#F08080','lightcyan':'#E0FFFF','lightgoldenrodyellow':'#FAFAD2','lightgray':'#D3D3D3','lightgrey':'#D3D3D3','lightgreen':'#90EE90',
		'lightpink':'#FFB6C1','lightsalmon':'#FFA07A','lightseagreen':'#20B2AA','lightskyblue':'#87CEFA','lightslategray':'#778899',
		'lightslategrey':'#778899','lightsteelblue':'#B0C4DE','lightyellow':'#FFFFE0','lime':'#00FF00','limegreen':'#32CD32','linen':'#FAF0E6',
		'magenta':'#FF00FF','maroon':'#800000','mediumaquamarine':'#66CDAA','mediumblue':'#0000CD','mediumorchid':'#BA55D3','mediumpurple':'#9370D8',
		'mediumseagreen':'#3CB371','mediumslateblue':'#7B68EE','mediumspringgreen':'#00FA9A','mediumturquoise':'#48D1CC','mediumvioletred':'#C71585',
		'midnightblue':'#191970','mintcream':'#F5FFFA','mistyrose':'#FFE4E1','moccasin':'#FFE4B5','navajowhite':'#FFDEAD','navy':'#000080','oldlace':'#FDF5E6',
		'olive':'#808000','olivedrab':'#6B8E23','orange':'#FFA500','orangered':'#FF4500','orchid':'#DA70D6','palegoldenrod':'#EEE8AA',
		'palegreen':'#98FB98','paleturquoise':'#AFEEEE','palevioletred':'#D87093','papayawhip':'#FFEFD5','peachpuff':'#FFDAB9','peru':'#CD853F',
		'pink':'#FFC0CB','plum':'#DDA0DD','powderblue':'#B0E0E6','purple':'#800080','red':'#FF0000','rosybrown':'#BC8F8F','royalblue':'#4169E1',
		'saddlebrown':'#8B4513','salmon':'#FA8072','sandybrown':'#F4A460','seagreen':'#2E8B57','seashell':'#FFF5EE','sienna':'#A0522D','silver':'#C0C0C0',
		'skyblue':'#87CEEB','slateblue':'#6A5ACD','slategray':'#708090','slategrey':'#708090','snow':'#FFFAFA','springgreen':'#00FF7F',
		'steelblue':'#4682B4','tan':'#D2B48C','teal':'#008080','thistle':'#D8BFD8','tomato':'#FF6347','turquoise':'#40E0D0','violet':'#EE82EE'
	};

	// helpers
	var absround = function(number){
		return (0.5 + number) << 0;
	};

	var hue2rgb = function(a, b, c) {  // http://www.w3.org/TR/css3-color/#hsl-color
		if(c < 0) c += 1;
		if(c > 1) c -= 1;
		if(c < 1/6) return a + (b - a) * 6 * c;
		if(c < 1/2) return b;
		if(c < 2/3) return a + (b - a) * (2/3 - c) * 6;
		return a;
	};

	var p2v = function(p){
		return isPercent.test(p) ? absround(parseInt(p) * 2.55) : p;
	};
	
	var isNamedColor = function(key){
		var lc = ('' + key).toLowerCase();
		return namedColors.hasOwnProperty(lc)
			? namedColors[lc]
			: null;
	};

	// patterns
	var isHex = /^#?([0-9a-f]{3}|[0-9a-f]{6})$/i;
	var isHSL = /^hsla?\((\d{1,3}?),\s*(\d{1,3}%),\s*(\d{1,3}%)(,\s*[01]?\.?\d*)?\)$/;
	var isRGB = /^rgba?\((\d{1,3}%?),\s*(\d{1,3}%?),\s*(\d{1,3}%?)(,\s*[01]?\.?\d*)?\)$/;
	var isPercent = /^\d+(\.\d+)*%$/;

	var hexBit = /([0-9a-f])/gi;
	var leadHex = /^#/;

	var matchHSL = /^hsla?\((\d{1,3}),\s*(\d{1,3})%,\s*(\d{1,3})%(,\s*([01]?\.?\d*))?\)$/;
	var matchRGB = /^rgba?\((\d{1,3}%?),\s*(\d{1,3}%?),\s*(\d{1,3}%?)(,\s*([01]?\.?\d*))?\)$/;

	/**
	* Color instance - get, update and output a Color between structures.
	* @constructor
	* @param {mixed} value Accepts any valid CSS color value e.g., #FF9900, rgb(255, 153, 0), rgba(100%, 40%, 0%, 0.8);
	* a hash with properties mapped to the Color instance e.g., red, green, saturation, brightness;
	* another Color instance; a numeric color value; a named CSS color
	* @class Instances of the Color class serve as abstract reprentations of the color itself, and don't need to be 
	* transformed from one format to another.  A single Color instance can have any component (red, green, blue, hue, saturation, lightness, brightness,
	* alpha) updated regardless of the source.  Further, all other components will be normalized automatically.  If a Color is instanced using a hex value,
	* it can have it's lightness component updated directly despite lightness being a HSL component.  Further, the same color instance can output it's
	* component values in any format without any extra conversions.  Conversion methods (getRGB, getHex) are provided just as helpers, and don't perform any
	* actual transformations.  They are not required for use or translation.  The standard component parts are available as instance methods - passing a value
	* argument to set, and each return the value as well (with or without setter arguments).  These components perform transformations and dispatch events, and
	* can be used without any sugar to manage the Color instance.
	* Component methods include:
	* .red()
	* .green()
	* .blue()
	* .hue()
	* .saturation()
	* .lightness()
	* .brightness()
	* .hex()
	* .decimal()
	* </ul>
	* @example
	* // instancing...
	* new Color();
	* new Color('#FF9900');
	* new Color(element.style.color);
	* new Color('pink');
	* new Color(123456);
	* new Color({ red : 255, green : 100, blue : 0 });
	* new Color(colorInstance);
	* // usage...
	* var color = new Color('#FF9900');
	* color.brightness(20);
	* element.style.backgroundColor = color;
	* console.log(color.getRGB());
	* console.log(color.saturation());
	*/
	function Color(value){

		this._listeners = {};

		this.subscribe(Events.RGB_UPDATED, this._RGBUpdated);
		this.subscribe(Events.HEX_UPDATED, this._HEXUpdated);
		this.subscribe(Events.HSL_UPDATED, this._HSLUpdated);
		this.subscribe(Events.HSV_UPDATED, this._HSVUpdated);
		this.subscribe(Events.INT_UPDATED, this._INTUpdated);
		
		this.parse(value);

	};

	Color.prototype._decimal = 0;  // 0 - 16777215
	Color.prototype._hex = '#000000';  // #000000 - #FFFFFF
	Color.prototype._red = 0;  // 0 - 255
	Color.prototype._green = 0;  // 0 - 255
	Color.prototype._blue = 0;  // 0 - 255
	Color.prototype._hue = 0;  // 0 - 360
	Color.prototype._saturation = 0;  // 0 - 100
	Color.prototype._lightness = 0;  // 0 - 100
	Color.prototype._brightness = 0;  // 0 - 100
	Color.prototype._alpha = 1;  // 0 - 1
	
	/**
	* Convert mixed variable to Color component properties, and adopt those properties.
	* @function
	* @param {mixed} value Accepts any valid CSS color value e.g., #FF9900, rgb(255, 153, 0), rgba(100%, 40%, 0%, 0.8);
	* a hash with properties mapped to the Color instance e.g., red, green, saturation, brightness;
	* another Color instance; a numeric color value; a named CSS color
	* @returns this
	* @example
	* var color = new Color();
	* color.parse();
	* color.parse('#FF9900');
	* color.parse(element.style.color);
	* color.parse('pink');
	* color.parse(123456);
	* color.parse({ red : 255, green : 100, blue : 0 });
	* color.parse(colorInstance);
	*/
	Color.prototype.parse = function(value){
		if(typeof value == 'undefined'){
			return this;
		};
		switch(true){
			case isFinite(value) :
				this.decimal(value);
				this.output = Color.INT;
				return this;
			case (value instanceof Color) :
				this.copy(value);
				return this;
			default : 
				switch(typeof value) {
					case 'object' :
						this.set(value);
						return this;
					case 'string' :
						switch(true){
							case (namedColors.hasOwnProperty(value)) :
								value = namedColors[value];
								var stripped = value.replace(leadHex, '');
								this.decimal(parseInt(stripped, 16));
								return this;
							case isHex.test(value) :
								var stripped = value.replace(leadHex, '');
								if(stripped.length == 3) {
									stripped = stripped.replace(hexBit, '$1$1');
								};
								this.decimal(parseInt(stripped, 16));
								return this;
							case isRGB.test(value) :
								var parts = value.match(matchRGB);
								this.red(p2v(parts[1]));
								this.green(p2v(parts[2]));
								this.blue(p2v(parts[3]));
								this.alpha(parseFloat(parts[5]) || 1);
								this.output = (isPercent.test(parts[1]) ? 2 : 1) + (parts[5] ? 2 : 0);
								return this;
							case isHSL.test(value) :  
								var parts = value.match(matchHSL);
								this.hue(parseInt(parts[1]));
								this.saturation(parseInt(parts[2]));
								this.lightness(parseInt(parts[3]));
								this.alpha(parseFloat(parts[5]) || 1);
								this.output = parts[5] ? 6: 5; 
								return this;
						};
				};		
			
		};
		return this;
	};

	/**
	* Create a duplicate of this Color instance
	* @function
	* @returns Color
	*/
	Color.prototype.clone = function(){
		return new Color(this.decimal());
	};

	/**
	* Copy values from another Color instance
	* @function
	* @param {Color} color Color instance to copy values from
	* @returns this
	*/
	Color.prototype.copy = function(color){
		return this.set(color.decimal());
	};

	/**
	* Set a color component value
	* @function
	* @param {string|object|number} key Name of the color component to defined, or a hash of key:value pairs, or a single numeric value
	* @param {string|number} value - Value of the color component to be set
	* @returns this
	* @example
	* var color = new Color();
	* color.set('lightness', 100);
	* color.set({ red : 255, green : 100 });
	* color.set(123456);
	*/
	Color.prototype.set = function(key, value){
		if(arguments.length == 1){
			if(typeof key == 'object'){
				for(var p in key){
					if(typeof this[p] == 'function'){
						this[p](key[p]);					
					};
				};
			} else if(isFinite(key)){
				this.decimal(key);
			}
		} else if(typeof this[key] == 'function'){
			this[key](value);
		};
		return this;
	};

	/**
	* sets the invoking Color instance component values to a point between the original value and the destination Color instance component value, multiplied by the factor
	* @function
	* @param {Color} destination Color instance to serve as the termination of the interpolation
	* @param {number} factor 0-1, where 0 is the origin Color and 1 is the destination Color, and 0.5 is halfway between.  This method will "blend" the colors.
	* @returns this
	* @example
	* var orange = new Color('#FF9900');
	* var white = new Color('#FFFFFF');
	* orange.interpolate(white, 0.5);
	*/
	Color.prototype.interpolate = function(destination, factor){
		if(!(destination instanceof Color)){
			destination = new Color(destination);
		};
		this._red = absround( +(this._red) + (destination._red - this._red) * factor );
		this._green = absround( +(this._green) + (destination._green - this._green) * factor );
		this._blue = absround( +(this._blue) + (destination._blue - this._blue) * factor );
		this._alpha = absround( +(this._alpha) + (destination._alpha - this._alpha) * factor );
		this.broadcast(Events.RGB_UPDATED);
		this.broadcast(Events.UPDATED);
		return this;
	};

	Color.prototype._RGB2HSL = function(){

		var r = this._red / 255;
		var g = this._green / 255;
		var b = this._blue / 255;

		var max = Math.max(r, g, b);
		var min = Math.min(r, g, b);
		var l = (max + min) / 2;
		var v = max;

		if(max == min) {
			this._hue = 0;
			this._saturation = 0;
			this._lightness = absround(l * 100);
			this._brightness = absround(v * 100);
			return;
		};

		var d = max - min;
		var s = d / ( ( l <= 0.5) ? (max + min) : (2 - max - min) );
		var h = ((max == r)
			? (g - b) / d + (g < b ? 6 : 0)
			: (max == g)
			 ? ((b - r) / d + 2)
			 : ((r - g) / d + 4)) / 6;

		this._hue = absround(h * 360);
		this._saturation = absround(s * 100);
		this._lightness = absround(l * 100);
		this._brightness = absround(v * 100);
	};
	Color.prototype._HSL2RGB = function(){
		var h = this._hue / 360;
		var s = this._saturation / 100;
		var l = this._lightness / 100;
		var q = l < 0.5	? l * (1 + s) : (l + s - l * s);
		var p = 2 * l - q;
		this._red = absround(hue2rgb(p, q, h + 1/3) * 255);
		this._green = absround(hue2rgb(p, q, h) * 255);
		this._blue = absround(hue2rgb(p, q, h - 1/3) * 255);
	};
	Color.prototype._HSV2RGB = function(){  // http://mjijackson.com/2008/02/rgb-to-hsl-and-rgb-to-hsv-color-model-conversion-algorithms-in-javascript
		var h = this._hue / 360;
		var s = this._saturation / 100;
		var v = this._brightness / 100;
		var r = 0;
		var g = 0;
		var b = 0;
		var i = Math.floor(h * 6);
		var f = h * 6 - i;
		var p = v * (1 - s);
		var q = v * (1 - f * s);
		var t = v * (1 - (1 - f) * s);
		switch(i % 6){
			case 0 :
				r = v, g = t, b = p;
				break;
			case 1 :
				r = q, g = v, b = p;
				break;
			case 2 :
				r = p, g = v, b = t;
				break;
			case 3 :
				r = p, g = q, b = v;
				break;
			case 4 :
				r = t, g = p, b = v
				break;
			case 5 :
				r = v, g = p, b = q;
				break;
		}
		this._red = absround(r * 255);
		this._green = absround(g * 255);
		this._blue = absround(b * 255);
	};
	Color.prototype._INT2HEX = function(){
		var x = this._decimal.toString(16);
		x = '000000'.substr(0, 6 - x.length) + x;
		this._hex = '#' + x.toUpperCase();
	};
	Color.prototype._INT2RGB = function(){
		this._red = this._decimal >> 16;
		this._green = (this._decimal >> 8) & 0xFF;
		this._blue = this._decimal & 0xFF;
	};
	Color.prototype._HEX2INT = function(){
		this._decimal = parseInt(this._hex, 16);
	};
	Color.prototype._RGB2INT = function(){
		this._decimal = (this._red << 16 | (this._green << 8) & 0xffff | this._blue);
	};


	Color.prototype._RGBUpdated = function(){
		this._RGB2INT();  // populate INT values
		this._RGB2HSL();  // populate HSL values
		this._INT2HEX();  // populate HEX values
	};
	Color.prototype._HSLUpdated = function(){
		this._HSL2RGB();  // populate RGB values
		this._RGB2INT();  // populate INT values
		this._INT2HEX();  // populate HEX values
	};
	Color.prototype._HSVUpdated = function(){
		this._HSV2RGB();  // populate RGB values
		this._RGB2INT();  // populate INT values
		this._INT2HEX();  // populate HEX values
	};
	Color.prototype._HEXUpdated = function(){
		this._HEX2INT();  // populate INT values
		this._INT2RGB();  // populate RGB values
		this._RGB2HSL();  // populate HSL values
	};
	Color.prototype._INTUpdated = function(){
		this._INT2RGB();  // populate RGB values
		this._RGB2HSL();  // populate HSL values
		this._INT2HEX();  // populate HEX values
	};

	Color.prototype._broadcastUpdate = function(){
		this.broadcast(Event.UPDATED);
	};

	/**
	* Set the decimal value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 (black) to 16777215 (white) - the decimal value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.decimal(123456);
	*/
	Color.prototype.decimal = function(value){
		return this._handle('_decimal', value, Events.INT_UPDATED);
	};

	/**
	* Set the hex value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {string} value Hex value to be set
	* @returns String
	* @example
	* var color = new Color();
	* color.hex('#FF9900');
	* color.hex('#CCC');
	*/
	Color.prototype.hex = function(value){
		return this._handle('_hex', value, Events.HEX_UPDATED);
	};

	/**
	* Set the red component value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 255 red component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.red(125);
	*/
	Color.prototype.red = function(value){
		return this._handle('_red', value, Events.RGB_UPDATED);
	};
	/**
	* Set the green component value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 255 green component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.green(125);
	*/
	Color.prototype.green = function(value){
		return this._handle('_green', value, Events.RGB_UPDATED);
	};
	/**
	* Set the blue component value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 255 blue component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.blue(125);
	*/
	Color.prototype.blue = function(value){
		return this._handle('_blue', value, Events.RGB_UPDATED);
	};

	/**
	* Set the hue component value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 360 hue component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.hue(280);
	*/
	Color.prototype.hue = function(value){
		return this._handle('_hue', value, Events.HSL_UPDATED);
	};
	
	/**
	* Set the saturation component value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 100 saturation component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.saturation(280);
	*/
	Color.prototype.saturation = function(value){
		return this._handle('_saturation', value, Events.HSL_UPDATED);
	};
	
	/**
	* Set the lightness component value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 100 lightness component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.lightness(80);
	*/	
	Color.prototype.lightness = function(value){
		return this._handle('_lightness', value, Events.HSL_UPDATED);
	};	
	
	/**
	* Set the brightness component value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 100 brightness component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.brightness(80);
	*/
	Color.prototype.brightness = function(value){
		return this._handle('_brightness', value, Events.HSV_UPDATED);
	};

	/**
	* Set the opacity value of the color, updates all other components, and dispatches Event.UPDATED
	* @function
	* @param {number} value 0 - 1 opacity component value to set
	* @returns Number
	* @example
	* var color = new Color();
	* color.alpha(0.5);
	*/
	Color.prototype.alpha = function(value){
		return this._handle('_alpha', value);
	};

	
	Color.prototype._handle = function(prop, value, event){
		if(typeof this[prop] != 'undefined'){
			if(typeof value != 'undefined'){
				if(value != this[prop]){
					this[prop] = value;
					if(event){
						this.broadcast(event);
					};
				};
				this.broadcast(Event.UPDATED);
			};
		};
		return this[prop];
	};
	
	/**
	* Returns a CSS-formatted hex string [e.g., #FF9900] from the Color's component values
	* @function
	* @returns String
	* @example
	* var color = new Color();
	* element.style.backgroundColor = color.getHex();
	*/
	Color.prototype.getHex = function(){
		return this._hex;
	};
	/**
	* Returns a CSS-formatted RGB string [e.g., rgb(255, 153, 0)] from the Color's component values
	* @function
	* @returns String
	* @example
	* var color = new Color();
	* element.style.backgroundColor = color.getRGB();
	*/
	Color.prototype.getRGB = function(){
		var components = [absround(this._red), absround(this._green), absround(this._blue)];
		return 'rgb(' + components.join(', ') + ')';
	};
	/**
	* Returns a CSS-formatted percentile RGB string [e.g., rgb(100%, 50%, 0)] from the Color's component values
	* @function
	* @returns String
	* @example
	* var color = new Color();
	* element.style.backgroundColor = color.getPRGB();
	*/
	Color.prototype.getPRGB = function(){
		var components = [absround(100 * this._red / 255) + '%', absround(100 * this._green / 255) + '%', absround(100 * this._blue / 255) + '%'];
		return 'rgb(' + components.join(', ') + ')';
	};
	/**
	* Returns a CSS-formatted RGBA string [e.g., rgba(255, 153, 0, 0.5)] from the Color's component values
	* @function
	* @returns String
	* @example
	* var color = new Color();
	* element.style.backgroundColor = color.getRGBA();
	*/
	Color.prototype.getRGBA = function(){
		var components = [absround(this._red), absround(this._green), absround(this._blue), this._alpha];
		return 'rgba(' + components.join(', ') + ')';
	};
	/**
	* Returns a CSS-formatted percentile RGBA string [e.g., rgba(100%, 50%, 0%, 0.5)] from the Color's component values
	* @function
	* @returns String
	* @example
	* var color = new Color();
	* element.style.backgroundColor = color.getPRGBA();
	*/
	Color.prototype.getPRGBA = function(){
		var components = [absround(100 * this._red / 255) + '%', absround(100 * this._green / 255) + '%', absround(100 * this._blue / 255) + '%', this._alpha];
		return 'rgba(' + components.join(', ') + ')';
	};
	/**
	* Returns a CSS-formatted HSL string [e.g., hsl(360, 100%, 100%)] from the Color's component values
	* @function
	* @returns String
	* @example
	* var color = new Color();
	* element.style.backgroundColor = color.getHSL();
	*/
	Color.prototype.getHSL = function(){
		var components = [absround(this._hue), absround(this._saturation) + '%', absround(this._lightness) + '%'];
		return 'hsl(' + components.join(', ') + ')';
	};
	/**
	* Returns a CSS-formatted HSLA string [e.g., hsl(360, 100%, 100%, 0.5)] from the Color's component values
	* @function
	* @returns String
	* @example
	* var color = new Color();
	* element.style.backgroundColor = color.getHSLA();
	*/
	Color.prototype.getHSLA = function(){
		var components = [absround(this._hue), absround(this._saturation) + '%', absround(this._lightness) + '%', this._alpha];
		return 'hsla(' + components.join(', ') + ')';
	};

	/**
	* Returns a tokenized string from the Color's component values
	* @function
	* @param {string} string The string to return, with tokens expressed as %token% that are replaced with component values.  Tokens are as follows:
	* r : red
	* g : green
	* b : blue
	* h : hue
	* s : saturation
	* l : lightness
	* v : brightness
	* a : alpha
	* x : hex
	* i : value
	* @returns String
	* @example
	* var color = new Color('#FF9900');
	* console.log(color.format('red=%r%, green=%g%, blue=%b%));
	*/
	Color.prototype.format = function(string){
		var tokens = {
			r : this._red,
			g : this._green,
			b : this._blue,
			h : this._hue,
			s : this._saturation,
			l : this._lightness,
			v : this._brightness,
			a : this._alpha,
			x : this._hex,
			d : this._decimal
		};
		for(var token in tokens){
			string = string.split('%' + token + '%').join(tokens[token]);
		};
		return string;
	};

	/**
	* Sets the format used by the native toString method
	* Color.HEX outputs #FF9900
	* Color.RGB outputs rgb(255, 153, 0)
	* Color.PRGB outputs rgb(100%, 50%, 0)
	* Color.RGBA outputs rgba(255, 153, 0, 0.5)
	* Color.PRGBA outputs rgba(100%, 50%, 0, 0.5)
	* Color.HSL outputs hsl(360, 100%, 80%)
	* Color.HSLA outputs hsla(360, 100%, 80%, 0.5)
	* @example
	* var color = new Color('#FF9900');
	* color.format = Color.RGB;
	* element.style.backgroundColor = color;
	* element.style.color = color;
	*/
	Color.prototype.output = 0;

	Color.HEX = 0;  // toString returns hex: #ABC123
	Color.RGB = 1;  // toString returns rgb: rgb(0, 100, 255)
	Color.PRGB = 2;  // toString returns percent rgb: rgb(0%, 40%, 100%)
	Color.RGBA = 3;  // toString returns rgba: rgba(0, 100, 255, 0.5)
	Color.PRGBA = 4;  // toString returns percent rgba: rgba(0%, 40%, 100%, 0.5)
	Color.HSL = 5;  // toString returns hsl: hsl(360, 50%, 50%)
	Color.HSLA = 6;  // toString returns hsla: hsla(360, 50%, 50%, 0.5)
	Color.INT = 7;  // toString returns decimal value

	Color.prototype.toString = function(){
		switch(this.output){
			case 0 :  // Color.HEX
				return this.getHex();
			case 1 :  // Color.RGB
				return this.getRGB();
			case 2 :  // Color.PRGB
				return this.getPRGB();
			case 3 :  // Color.RGBA
				return this.getRGBA();
			case 4 :  // Color.PRGBA
				return this.getPRGBA();
			case 5 :  // Color.HSL
				return this.getHSL();
			case 6 :  // Color.HSLA
				return this.getHSLA();
			case 7 :  // Color.INT
				return this._decimal;
		};
		return this.getHex();
	};

	// Event Management
	Color.prototype._listeners = null;
	Color.prototype._isSubscribed = function(type){
		return this._listeners[type] != null;
	};
	/**
	* @function
	* @param {string} type Event type to listen for
	* @param {function} callback listener to register to the event
	* @example 
	* var color = new Color();
	* color.subscribe(Color.Event.UPDATED, function(){
	*   alert('this color has been updated');
	* });
	* color.red(255);
	*/
	Color.prototype.subscribe = function(type, callback){
		if(!this._isSubscribed(type)) {
			this._listeners[type] = [];
		};
		this._listeners[type].push(callback);
	};
	
	/**
	* @function
	* @param {string} type Event type to remove the listener from
	* @param {function} callback listener to unregister from the event
	* @example 
	* var color = new Color();
	* var handler = function(){
	*   console.log('this color has been updated');
	* });
	* color.subscribe(Color.Event.UPDATED, handler);
	* color.red(255);
	* color.unsubscribe(Color.Event.UPDATED, handler);
	* color.red(0);
	*/
	Color.prototype.unsubscribe = function(type, callback){
		if(!this._isSubscribed(type)) {
			return;
		};
		var stack = this._listeners[type];
		for(var i = 0, l = stack.length; i < l; i++){
			if(stack[i] === callback){
				stack.splice(i, 1);
				return this.unsubscribe(type, callback);
			};
		};
	};
	
	/**
	* @function
	* @param {string} type Event type to dispatch
	* @param {array} params Array of arguments to pass to listener
	* @example 
	* var color = new Color();
	* var handler = function(a, b){
	*   console.log('a=' + a, 'b=' + b);
	* });
	* color.subscribe('arbitraryEvent', handler);
	* color.broadcast('arbitraryEvent', ['A', 'B']);
	*/
	Color.prototype.broadcast = function(type, params){
		if(!this._isSubscribed(type)) {
			return;
		}
		var stack = this._listeners[type];
		var l = stack.length;
		for(var i = 0; i < l; i++) {
			stack[i].apply(this, params);
		}
	};
	
	/**
	* Blends the color from it's current state to the target Color over the duration
	* @function
	* @param {number} duration duration of tween in millisecond
	* @param {Color} color destination color
	* @returns number
	* @example 
	* var color = new Color('#FF9900');
	* color.tween(2000, '#FFFFFF');
	*/
	Color.prototype.tween = function(duration, color){
		if(!(color instanceof Color)){
			color = new Color(color);
		};
		var start = +(new Date());
		var ref = this;
		this.broadcast('tweenStart');
		var interval = setInterval(function(){
			var ellapsed = +(new Date()) - start;
			var delta = Math.min(1, ellapsed / duration);
			ref.interpolate(color, delta);
			ref.broadcast('tweenProgress');
			if(delta == 1){
				clearInterval(interval);
				ref.broadcast('tweenComplete');
			};
		}, 20);
		return interval;  // return so it can be cancelled early
	};
	
	/**
	* Binds the color to an object's property - whenever the Color is updated, the property will be set to the value of the Color instance's toString method
	* @function
	* @param {object} object Object containing the property to update
	* @param {string} property Name of the property to update
	* @example 
	* var color = new Color('#FF9900');
	* color.bind(someElement.style, 'backgroundColor');
	*/
	Color.prototype.bind = function(object, property){
		var ref = this;
		this.subscribe('updated', function(){
			object[property] = ref.toString();
		});
	};
	
	
	/**
	* [static] Returns a Color instance of a random color
	* @function
	* @returns Color
	* @example 
	* var gray = Color.interpolate('#FFFFFF', '#000000', 0.5);
	*/
	Color.random = function(){
		return new Color(absround(Math.random() * 16777215));
	};
	
	
	/**
	* [static] Returns a Color instance bound to an object's property, set to the value of the property
	* @function
	* @param {object} object Object containing the property to update
	* @param {string} property Name of the property to update
	* @returns Color
	* @example 
	* var color = Color.bind(someElement.style, 'backgroundColor');
	*/
	Color.bind = function(object, property){
		var color = new Color(object[property]);
		color.bind(object, property);
		return color;
	};
	
	Color.Events = Events;
	
	if (typeof define === 'function') {
		define('Color', [], function() {
			return Color;
		});
	};
	
	return Color;	
	
})(window);

	
	
	



/*
--------------------------------------------------------------
FUNCTIONS
--------------------------------------------------------------
*/
function responsive_resize(){
 	var current_width = $(window).width();
    if(current_width < 768){
      $('html').addClass("mobile").removeClass("desktop").removeClass("m320").removeClass("m768").removeClass("tablet");
 	}else if (current_width < 1023){
      $('html').addClass("tablet").removeClass("desktop").removeClass("m320").removeClass("m768").removeClass("mobile");
 	}else if (current_width > 1023){
      $('html').addClass("desktop").removeClass("m320").removeClass("m768").removeClass("tablet").removeClass("mobile");
  	}
}

// seeded random number generator
!function(a,b,c,d,e,f,g,h,i){function j(a){var b,c=a.length,e=this,f=0,g=e.i=e.j=0,h=e.S=[];for(c||(a=[c++]);d>f;)h[f]=f++;for(f=0;d>f;f++)h[f]=h[g=t&g+a[f%c]+(b=h[f])],h[g]=b;(e.g=function(a){for(var b,c=0,f=e.i,g=e.j,h=e.S;a--;)b=h[f=t&f+1],c=c*d+h[t&(h[f]=h[g=t&g+b])+(h[g]=b)];return e.i=f,e.j=g,c})(d)}function k(a,b){return b.i=a.i,b.j=a.j,b.S=a.S.slice(),b}function l(a,b){var c,d=[],e=typeof a;if(b&&"object"==e)for(c in a)try{d.push(l(a[c],b-1))}catch(f){}return d.length?d:"string"==e?a:a+"\0"}function m(a,b){for(var c,d=a+"",e=0;e<d.length;)b[t&e]=t&(c^=19*b[t&e])+d.charCodeAt(e++);return o(b)}function n(c){try{return p?o(p.randomBytes(d)):(a.crypto.getRandomValues(c=new Uint8Array(d)),o(c))}catch(e){return[+new Date,a,(c=a.navigator)&&c.plugins,a.screen,o(b)]}}function o(a){return String.fromCharCode.apply(0,a)}var p,q=c.pow(d,e),r=c.pow(2,f),s=2*r,t=d-1,u=c["seed"+i]=function(a,f,g){var h=[];f=1==f?{entropy:!0}:f||{};var p=m(l(f.entropy?[a,o(b)]:null==a?n():a,3),h),t=new j(h);return m(o(t.S),b),(f.pass||g||function(a,b,d,e){return e&&(e.S&&k(e,t),a.state=function(){return k(t,{})}),d?(c[i]=a,b):a})(function(){for(var a=t.g(e),b=q,c=0;r>a;)a=(a+c)*d,b*=d,c=t.g(1);for(;a>=s;)a/=2,b/=2,c>>>=1;return(a+c)/b},p,"global"in f?f.global:this==c,f.state)};if(m(c[i](),b),g&&g.exports){g.exports=u;try{p=require("crypto")}catch(v){}}else h&&h.amd&&h(function(){return u})}(this,[],Math,256,6,52,"object"==typeof module&&module,"function"==typeof define&&define,"random");